#ifndef PQ_H
#define PQ_H

#include <iostream>
#include <algorithm>
#include <stdexcept>
#include "Node.h"

using namespace std;

template <typename T>
class PQ {
private:
    Node<T>* head;
    Node<T>* tail;
    int count;

    Node<T>* getNode(int n);
    void bubbleUp(Node<T>* node);
    void bubbleDown(Node<T>* node);

public:
    PQ();
    ~PQ();

    bool empty() const;
    int size() const;
    void push(T value);
    T pop();
};

// Implementation

template <typename T>
Node<T>* PQ<T>::getNode(int n) {
    if (n <= 0 || n > count) return nullptr;
    if (n == 1) return head;

    // 计算最高位的位置（不包括根的最高位），然后从高位到低位依次检查位
    int temp = n;
    int msb = 0; // msb 为从根到目标需要的步数
    while (temp > 1) {
        temp /= 2;
        msb++;
    }

    Node<T>* curr = head;
    for (int i = msb - 1; i >= 0; --i) {
        int bit = (n >> i) & 1; // 0 -> right, 1 -> left
        if (bit == 0) {
            curr = curr->right;
        } else {
            curr = curr->left;
        }
        if (!curr) return nullptr;
    }
    return curr;
}

template <typename T>
void PQ<T>::bubbleUp(Node<T>* node) {
    while (node->parent && node->value < node->parent->value) {
        swap(node->value, node->parent->value);
        node = node->parent;
    }
}

template <typename T>
void PQ<T>::bubbleDown(Node<T>* node) {
    while (true) {
        Node<T>* smallest = node;
        // Check Right child first as per structure preference? 
        // Order doesn't strictly matter for correctness of MinHeap, 
        // but we should check both children.
        if (node->right && node->right->value < smallest->value) {
            smallest = node->right;
        }
        if (node->left && node->left->value < smallest->value) {
            smallest = node->left;
        }

        if (smallest != node) {
            swap(node->value, smallest->value);
            node = smallest;
        } else {
            break;
        }
    }
}

template <typename T>
PQ<T>::PQ() : head(nullptr), tail(nullptr), count(0) {}

template <typename T>
PQ<T>::~PQ() {
    while (!empty()) {
        pop();
    }
}

template <typename T>
bool PQ<T>::empty() const {
    return count == 0;
}

template <typename T>
int PQ<T>::size() const {
    return count;
}

template <typename T>
void PQ<T>::push(T value) {
    Node<T>* newNode = new Node<T>(value);
    if (count == 0) {
        head = tail = newNode;
        count++;
        return;
    }

    count++;
    Node<T>* parent = getNode(count / 2);
    
    newNode->parent = parent;
    // Even -> Right child (0), Odd -> Left child (1)
    if (count % 2 == 0) {
        parent->right = newNode;
    } else {
        parent->left = newNode;
    }
    
    tail = newNode;
    bubbleUp(tail);
}

template <typename T>
T PQ<T>::pop() {
    if (empty()) {
        throw runtime_error("PQ is empty");
    }

    T ret = head->value;

    if (count == 1) {
        delete head;
        head = tail = nullptr;
        count = 0;
        return ret;
    }

    head->value = tail->value;

    Node<T>* oldTail = tail;
    Node<T>* parent = oldTail->parent;
    
    if (parent) {
        if (parent->right == oldTail) parent->right = nullptr;
        else parent->left = nullptr;
    }

    delete oldTail;
    count--;

    tail = getNode(count);
    bubbleDown(head);

    return ret;
}

#endif
