#include "BigInt.h"
#include <cstring>
#include <algorithm>
using namespace std;

static bool is_valid_digits(const char* str) {
    if (!str || *str == '\0') return false;
    // allow leading zeros, but must be digits only
    for (const char* p = str; *p; ++p) {
        if (*p < '0' || *p > '9') return false;
    }
    return true;
}

void BigInt::init_with_zero() {
    isZero = true;
    len = 1;
    size = 1;
    s = new char[size];
    s[0] = '0';
}

void BigInt::ensure_capacity(size_t cap) {
    if (cap <= size) return;
    size_t newCap = max(cap, size * 2);
    char* ns = new char[newCap];
    memcpy(ns, s, len);
    delete[] s;
    s = ns;
    size = newCap;
}

void BigInt::normalize() {
    // remove leading zeros from s (MSD at s[0])
    size_t i = 0;
    while (i < len && s[i] == '0') ++i;
    if (i == len) {
        // all zeros -> represent as single '0'
        isZero = true;
        len = 1;
        s[0] = '0';
        return;
    }
    if (i > 0) {
        // shift left by i positions
        size_t newLen = len - i;
        for (size_t j = 0; j < newLen; ++j) s[j] = s[j + i];
        len = newLen;
    }
    isZero = false;
}

BigInt::BigInt(size_t value) {
    if (value == 0) {
        init_with_zero();
        return;
    }
    isZero = false;
    // compute number of digits
    size_t v = value;
    size_t digits = 0;
    while (v) { v /= 10; ++digits; }
    len = digits;
    size = max<size_t>(digits, 1);
    s = new char[size];
    // fill from end
    size_t idx = len;
    v = value;
    while (v) {
        s[--idx] = char('0' + (v % 10));
        v /= 10;
    }
}

BigInt::BigInt(const char* str) {
    if (!is_valid_digits(str)) {
        // treat invalid input as zero
        init_with_zero();
        return;
    }
    len = strlen(str);
    size = max<size_t>(len, 1);
    s = new char[size];
    memcpy(s, str, len);
    normalize();
}

BigInt::BigInt(const BigInt& other) : isZero(other.isZero), len(other.len), size(other.size) {
    s = new char[size];
    memcpy(s, other.s, len);
}

BigInt::~BigInt() {
    delete[] s;
}

BigInt& BigInt::operator=(const BigInt& other) {
    if (this == &other) return *this;
    isZero = other.isZero;
    len = other.len;
    if (other.size > size) {
        delete[] s;
        size = other.size;
        s = new char[size];
    }
    memcpy(s, other.s, len);
    return *this;
}

BigInt BigInt::operator+(const BigInt& rhs) const {
    if (isZero) return rhs;
    if (rhs.isZero) return *this;
    // add from LSD: since MSD at s[0], we iterate from back
    size_t i = len;
    size_t j = rhs.len;
    size_t maxLen = max(len, rhs.len);
    BigInt result(static_cast<size_t>(0));
    result.ensure_capacity(maxLen + 1);
        result.len = maxLen + 1; // possible carry
    result.isZero = false;
    // temp fill zeros
    for (size_t k = 0; k < result.len; ++k) result.s[k] = '0';

    int carry = 0;
    size_t ri = result.len;
    while (i > 0 || j > 0 || carry) {
        int a = (i > 0) ? (s[--i] - '0') : 0;
        int b = (j > 0) ? (rhs.s[--j] - '0') : 0;
        int sum = a + b + carry;
        carry = sum / 10;
        int digit = sum % 10;
        result.s[--ri] = char('0' + digit);
    }
    // if any leading (unused) positions remain, keep '0' there
    // normalize to remove leading zeros
    result.normalize();
    return result;
}

BigInt& BigInt::operator+=(const BigInt& rhs) {
    *this = *this + rhs;
    return *this;
}

BigInt BigInt::operator*(const BigInt& rhs) const {
    if (isZero || rhs.isZero) {
           return BigInt(static_cast<size_t>(0));
    }
    // result length at most len + rhs.len
        BigInt result(static_cast<size_t>(0));
    result.ensure_capacity(len + rhs.len);
    result.len = len + rhs.len;
    result.isZero = false;
    for (size_t i = 0; i < result.len; ++i) result.s[i] = '0';

    // multiply from LSD
    // indices: our LSD at s[len-1], rhs LSD at rhs.s[rhs.len-1]
    for (size_t i = 0; i < len; ++i) {
        int carry = 0;
        int a = s[len - 1 - i] - '0';
        size_t pos = result.len - 1 - i; // starting position in result
        for (size_t j = 0; j < rhs.len; ++j) {
            int b = rhs.s[rhs.len - 1 - j] - '0';
            int cur = (result.s[pos - j] - '0') + a * b + carry;
            carry = cur / 10;
            result.s[pos - j] = char('0' + (cur % 10));
        }
        // place remaining carry
        size_t k = pos - rhs.len;
        while (carry > 0) {
            int cur = (result.s[k] - '0') + carry;
            carry = cur / 10;
            result.s[k] = char('0' + (cur % 10));
            if (k == 0) break; else --k;
        }
    }
    result.normalize();
    return result;
}

BigInt& BigInt::operator*=(const BigInt& rhs) {
    *this = *this * rhs;
    return *this;
}

bool BigInt::operator==(const BigInt& rhs) const {
    if (isZero && rhs.isZero) return true;
    if (len != rhs.len) return false;
    for (size_t i = 0; i < len; ++i) {
        if (s[i] != rhs.s[i]) return false;
    }
    return true;
}

bool BigInt::operator!=(const BigInt& rhs) const { return !(*this == rhs); }

bool BigInt::operator<(const BigInt& rhs) const {
    if (isZero && rhs.isZero) return false;
    if (isZero && !rhs.isZero) return true;
    if (!isZero && rhs.isZero) return false;
    if (len != rhs.len) return len < rhs.len;
    for (size_t i = 0; i < len; ++i) {
        if (s[i] != rhs.s[i]) return s[i] < rhs.s[i];
    }
    return false; // equal
}

bool BigInt::operator<=(const BigInt& rhs) const { return (*this < rhs) || (*this == rhs); }
bool BigInt::operator>(const BigInt& rhs) const { return rhs < *this; }
bool BigInt::operator>=(const BigInt& rhs) const { return !(*this < rhs); }

ostream& operator<<(ostream& os, const BigInt& x) {
    os.write(x.s, static_cast<streamsize>(x.len));
    return os;
}